/*========================================================================*
 |                      queryproc.pro
 |      Copyright (c) 2020 Applied Logic Systems, Inc.
 |              -- Utilities for processing cgi queries
 *========================================================================*/

/*!---------------------------------------------------------------------
 |      split_comps/2
 |	split_comps(QS, Result)
 |	split_comps(+, -)
 |
 |      - splits a cgi query into a list of equality statements
 |
 |	If QS is an atom representing the query string of a POST or
 |	GET cgi query, split_comps/2 breaks that into a list of 
 |	expressions of the form A=B by converting QS to a list of
 |	character codes, and involking break_comps/2.
 *!--------------------------------------------------------------------*/
split_comps(QS, Result)
	:-
	atom_codes(QS, QChars),
	break_comps(QChars, Result).

/*!---------------------------------------------------------------------
 |	break_comps/2
 |	break_comps(QChars, EqsList)
 |	break_comps(+, -)
 |
 |	- breaks a list of char codes into a list of equality statements
 |
 |	If QChars is list of character codes corresponding to the query
 |	string atom representing the query string of a POST or GET cgi
 |	break_comps/2 breaks that into a list of expressions of the 
 |	form A=B.
 *!--------------------------------------------------------------------*/
break_comps([], []).
break_comps(QChars, [Left=Right | Result])
	:-
        asplit0(QChars, 0'&, FirstCs, RestCs),
	!,
	asplit0(FirstCs, 0'=, LeftCs, RightCs),
	atom_codes(Left, LeftCs),
	decode(RightCs, DecodedRight0Cs, Qtn),
	(Qtn ->
		append([0'', 0'', 0'' | DecodedRight0Cs], [0'', 0'', 0''], DecodedRightCs)
		;
		DecodedRightCs = DecodedRight0Cs
	),
	atom_codes(Right, DecodedRightCs),
	
	break_comps(RestCs, Result).
break_comps(QChars, [Left=Right])
	:-
	asplit0(QChars, 0'=, LeftCs, RightCs),
	atom_codes(Left, LeftCs),
	decode(RightCs, DecodedRight0Cs, Qtn),
	(Qtn ->
		append([0'', 0'', 0'' | DecodedRight0Cs], [0'', 0'', 0''], DecodedRightCs)
		;
		DecodedRightCs = DecodedRight0Cs
	),
	atom_codes(Right, DecodedRightCs).






/*!---------------------------------------------------------------------
 |	decode/3
 |	decode(CharCodeListIn, OutCharCodeList, QuasiBool)
 |	decode(+, -, -)
 |
 |	- decodes strings of char codes containing URI encodings
 |
 |	If CharCodeListIn is a list of character codes, then
 |	OutCharCodeList is the result of replacing all uri encodings
 |		%k1k2
 |	occurring in CharCodeListIn by the character code of the
 |	character uri encoded by %k1k2.
 *!--------------------------------------------------------------------*/
/* Decodes strings */

decode([], [], Qtn) :-
	(var(Qtn) -> Qtn = fail ; true).

decode([0'%, N1, N2 | RestInChars], [Decoded | RestOutChars], true)
	:-!,
	uricode(N1, N2, Decoded),
	decode(RestInChars, RestOutChars, true).

decode([Char | RestInChars], [Char | RestOutChars], Qtn)
	:-
	decode(RestInChars, RestOutChars, Qtn).


/*!---------------------------------------------------------------------
 |	uricode/3
 |	uricode(N1, N2, EncodedChar)
 |	uricode(+, +, +)
 |
 |	- maps URI encoding of reserved chars
 |
 |	If C is an HTML reserved character, if CC is its (prolog)
 |	character code, and if %k1k2 is it's URI encoding, then
 |	the fact uricode(N1, N2, CC) is true, where "k1k2" = [N1,N2].
 |	Used by decode/3.
 *!--------------------------------------------------------------------*/
uricode(50, 48, 0' ).	%% coded: %20
uricode(50, 49, 0'!).	%% coded: %21
uricode(50, 51, 0'#).	%% coded: %23
uricode(50, 52, 0'$).	%% coded: %24
uricode(50, 53, 0'%).	%% coded: %25
uricode(50, 54, 0'&).	%% coded: %26
uricode(50, 55, 0'').	%% coded: %27
uricode(50, 56, 0'().	%% coded: %28
uricode(50, 57, 0')).	%% coded: %29
uricode(50, 65, 0'*).	%% coded: %2A
uricode(50, 66, 0'+).	%% coded: %2B
uricode(50, 67, 0',).	%% coded: %2C
uricode(50, 70, 0'/).	%% coded: %2F
uricode(51, 65, 0':).	%% coded: %3A
uricode(51, 66, 0';).	%% coded: %3B
uricode(51, 67, 0'=).	%% coded: %3D
uricode(51, 70, 0'?).	%% coded: %3F
uricode(52, 48, 0'@).	%% coded: %40
uricode(53, 66, 0'[).	%% coded: %5B
uricode(53, 68, 0']).	%% coded: %5D

proc_qs(Result) 
	:-
	getenv('QUERY_STRING', QS),
	split_comps(QS, Result).


tq :-
	Qs = 'loginid=asdfasdf&email=ken%40desert-light.org&pwdfield=asdf&pwdfield2=asdf',
	split_comps(Qs, Result),
	write(r=Result),nl.
th :-
	Qs = 'loginid=zappo&email=silly%40zing-light.org&pwdfield=asdf&pwdfield2=asdf',
	split_comps(Qs, Result),
	write(r=Result),nl.

